<?php

/**
 * DeleteController.php
 * Copyright (c) 2020 james@firefly-iii.org
 *
 * This file is part of Firefly III (https://github.com/firefly-iii).
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

declare(strict_types=1);

namespace FireflyIII\Http\Controllers\Bill;

use FireflyIII\Support\Facades\Preferences;
use FireflyIII\Http\Controllers\Controller;
use FireflyIII\Models\Bill;
use FireflyIII\Repositories\Bill\BillRepositoryInterface;
use Illuminate\Contracts\View\Factory;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Routing\Redirector;
use Illuminate\View\View;

/**
 * Class DeleteController
 */
class DeleteController extends Controller
{
    private BillRepositoryInterface $repository;

    /**
     * BillController constructor.
     */
    public function __construct()
    {
        parent::__construct();

        app('view')->share('showBudget', true);

        $this->middleware(
            function ($request, $next) {
                app('view')->share('title', (string) trans('firefly.bills'));
                app('view')->share('mainTitleIcon', 'fa-calendar-o');
                $this->repository = app(BillRepositoryInterface::class);

                return $next($request);
            }
        );
    }

    /**
     * Delete a bill.
     *
     * @return Factory|View
     */
    public function delete(Bill $bill): Factory|\Illuminate\Contracts\View\View
    {
        // put previous url in session
        $this->rememberPreviousUrl('bills.delete.url');
        $subTitle = (string) trans('firefly.delete_bill', ['name' => $bill->name]);

        return view('bills.delete', ['bill' => $bill, 'subTitle' => $subTitle]);
    }

    /**
     * Destroy a bill.
     */
    public function destroy(Request $request, Bill $bill): Redirector|RedirectResponse
    {
        $name = $bill->name;
        $this->repository->destroy($bill);

        $request->session()->flash('success', (string) trans('firefly.deleted_bill', ['name' => $name]));
        Preferences::mark();

        return redirect($this->getPreviousUrl('bills.delete.url'));
    }
}
