<?php
/**
 * Widget API: WP_Widget_Media_Video class
 *
 * @package ClassicPress
 * @subpackage Widgets
 * @since 4.8.0
 */

/**
 * Core class that implements a video widget.
 *
 * @since 4.8.0
 *
 * @see WP_Widget_Media
 * @see WP_Widget
 */
class WP_Widget_Media_Video extends WP_Widget_Media {

	/**
	 * Constructor.
	 *
	 * @since 4.8.0
	 */
	public function __construct() {
		parent::__construct(
			'media_video',
			__( 'Video' ),
			array(
				'description' => __( 'Displays a video from the media library or from YouTube, Vimeo, or another provider.' ),
				'mime_type'   => 'video',
			)
		);
	}

	/**
	 * Get schema for properties of a widget instance (item).
	 *
	 * @since 4.8.0
	 *
	 * @see WP_REST_Controller::get_item_schema()
	 * @see WP_REST_Controller::get_additional_fields()
	 * @link https://core.trac.wordpress.org/ticket/35574
	 *
	 * @return array Schema for properties.
	 */
	public function get_instance_schema() {

		$schema = array(
			'preload' => array(
				'type'                  => 'string',
				'enum'                  => array( 'none', 'auto', 'metadata' ),
				'default'               => 'metadata',
				'description'           => __( 'Preload' ),
				'should_preview_update' => false,
			),
			'loop'    => array(
				'type'                  => 'boolean',
				'default'               => false,
				'description'           => __( 'Loop' ),
				'should_preview_update' => false,
			),
			'content' => array(
				'type'                  => 'string',
				'default'               => '',
				'sanitize_callback'     => 'wp_kses_post',
				'description'           => __( 'Tracks (subtitles, captions, descriptions, chapters, or metadata)' ),
				'should_preview_update' => false,
			),
		);

		foreach ( wp_get_video_extensions() as $video_extension ) {
			$schema[ $video_extension ] = array(
				'type'        => 'string',
				'default'     => '',
				'format'      => 'uri',
				/* translators: %s: Video extension. */
				'description' => sprintf( __( 'URL to the %s video source file' ), $video_extension ),
			);
		}

		return array_merge( $schema, parent::get_instance_schema() );
	}

	/**
	 * Render the media on the frontend.
	 *
	 * @since 4.8.0
	 *
	 * @param array $instance Widget instance props.
	 */
	public function render_media( $instance ) {
		$instance   = array_merge( wp_list_pluck( $this->get_instance_schema(), 'default' ), $instance );
		$attachment = null;

		if ( $this->is_attachment_with_mime_type( $instance['attachment_id'], $this->widget_options['mime_type'] ) ) {
			$attachment = get_post( $instance['attachment_id'] );
		}

		$src = $instance['url'];
		if ( $attachment ) {
			$src = wp_get_attachment_url( $attachment->ID );
		}

		if ( empty( $src ) ) {
			return;
		}

		$youtube_pattern = '#^https?://(?:www\.)?(?:youtube\.com/watch|youtu\.be/)#';
		$vimeo_pattern   = '#^https?://(.+\.)?vimeo\.com/.*#';

		if ( $attachment || preg_match( $youtube_pattern, $src ) || preg_match( $vimeo_pattern, $src ) ) {
			add_filter( 'wp_video_shortcode', array( $this, 'inject_video_max_width_style' ) );

			echo wp_video_shortcode(
				array_merge(
					$instance,
					compact( 'src' )
				),
				$instance['content']
			);

			remove_filter( 'wp_video_shortcode', array( $this, 'inject_video_max_width_style' ) );
		} else {
			echo $this->inject_video_max_width_style( wp_oembed_get( $src ) );
		}
	}

	/**
	 * Inject max-width and remove height for videos too constrained to fit inside sidebars on frontend.
	 *
	 * @since 4.8.0
	 *
	 * @param string $html Video shortcode HTML output.
	 * @return string HTML Output.
	 */
	public function inject_video_max_width_style( $html ) {
		$html = preg_replace( '/\sheight="\d+"/', '', $html );
		$html = preg_replace( '/\swidth="\d+"/', '', $html );
		$html = preg_replace( '/(?<=width:)\s*\d+px(?=;?)/', '100%', $html );
		return $html;
	}

	/**
	 * Outputs the settings update form.
	 *
	 * Now renders immediately with PHP instead of just-in-time JavaScript
	 *
	 * @since CP-2.5.0
	 *
	 * @param array $instance Current settings.
	 */
	public function form( $instance ) {
		$title         = ! empty( $instance['title'] ) ? $instance['title'] : '';
		$attachment_id = ! empty( $instance['attachment_id'] ) ? $instance['attachment_id'] : 0;
		$url           = ! empty( $instance['url'] ) ? $instance['url'] : '';
		$preload       = ! empty( $instance['preload'] ) ? $instance['preload'] : 'metadata';
		$loop          = ! empty( $instance['loop'] ) ? true : false;
		$content       = ! empty( $instance['content'] ) ? $instance['content'] : '';
		$mp4           = ! empty( $instance['mp4'] ) ? $instance['mp4'] : '';
		$m4v           = ! empty( $instance['m4v'] ) ? $instance['m4v'] : '';
		$webm          = ! empty( $instance['webm'] ) ? $instance['webm'] : '';
		$ogv           = ! empty( $instance['ogv'] ) ? $instance['ogv'] : '';
		$flv           = ! empty( $instance['flv'] ) ? $instance['flv'] : '';
		$mov           = ! empty( $instance['mov'] ) ? $instance['mov'] : '';
		$quicktime     = ! empty( $instance['quicktime'] ) ? $instance['quicktime'] : '';
		$nonce         = wp_create_nonce( 'video_editor-' . $attachment_id );

		if ( $url === '' ) {
			if ( $attachment_id ) {
				$url = wp_get_attachment_url( $attachment_id );
			} elseif ( $mp4 ) {
				$url = $mp4;
			} elseif ( $m4v ) {
				$url = $m4v;
			} elseif ( $webm ) {
				$url = $webm;
			} elseif ( $ogv ) {
				$url = $ogv;
			} elseif ( $flv ) {
				$url = $flv;
			} elseif ( $mov ) {
				$url = $mov;
			} elseif ( $quicktime ) {
				$url = quicktime;
			}
		}
		?>

		<div class="media-widget-control selected">
			<fieldset>
				<label for="<?php echo esc_attr( $this->get_field_id( 'title' ) ); ?>"><?php esc_html_e( 'Title:' ); ?></label>
				<input id="<?php echo esc_attr( $this->get_field_id( 'title' ) ); ?>" name="<?php echo esc_attr( $this->get_field_name( 'title' ) ); ?>" type="text" class="widefat" value="<?php echo esc_attr( $title ); ?>">
			</fieldset>

			<?php
			if ( $url && file_exists( get_attached_file( $attachment_id ) ) ) {
				?>

				<div class="media-widget-preview media_video populated"><?php echo wp_video_shortcode( array( 'src' => $url ) ); ?></div>

				<fieldset class="media-widget-buttons">
					<button type="button" class="button edit-media selected" data-edit-nonce="<?php echo esc_attr( $nonce ); ?>"><?php esc_html_e( 'Edit Video' ); ?></button>
					<button type="button" class="button change-media select-media"><?php esc_html_e( 'Replace Video' ); ?></button>
				</fieldset>

				<?php
			} else {
				?>

				<fieldset class="media-widget-preview media_video">
					<div class="attachment-media-view">
						<button type="button" class="select-media button-add-media"><?php esc_html_e( 'Add Video' ); ?></button>
					</div>
				</fieldset>

				<?php
			}
			?>

			<input id="<?php echo esc_attr( $this->get_field_id( 'preload' ) ); ?>" name="<?php echo esc_attr( $this->get_field_name( 'preload' ) ); ?>" type="hidden" data-property="preload" class="media-widget-instance-property" value="<?php echo esc_attr( esc_attr( $preload ) ); ?>">
			<input id="<?php echo esc_attr( $this->get_field_id( 'loop' ) ); ?>" name="<?php echo esc_attr( $this->get_field_name( 'loop' ) ); ?>" type="hidden" data-property="loop" class="media-widget-instance-property" value="<?php echo esc_attr( $loop ); ?>">
			<input id="<?php echo esc_attr( $this->get_field_id( 'content' ) ); ?>" name="<?php echo esc_attr( $this->get_field_name( 'content' ) ); ?>" type="hidden" data-property="content" class="media-widget-instance-property" value="<?php echo esc_attr( $content ); ?>">
			<input id="<?php echo esc_attr( $this->get_field_id( 'mp4' ) ); ?>" name="<?php echo esc_attr( $this->get_field_name( 'mp4' ) ); ?>" type="hidden" data-property="mp4" class="media-widget-instance-property" value="<?php echo esc_attr( $mp4 ); ?>">
			<input id="<?php echo esc_attr( $this->get_field_id( 'm4v' ) ); ?>" name="<?php echo esc_attr( $this->get_field_name( 'm4v' ) ); ?>" type="hidden" data-property="m4v" class="media-widget-instance-property" value="<?php echo esc_attr( $m4v ); ?>">
			<input id="<?php echo esc_attr( $this->get_field_id( 'webm' ) ); ?>" name="<?php echo esc_attr( $this->get_field_name( 'webm' ) ); ?>" type="hidden" data-property="webm" class="media-widget-instance-property" value="<?php echo esc_attr( $webm ); ?>">
			<input id="<?php echo esc_attr( $this->get_field_id( 'ogv' ) ); ?>" name="<?php echo esc_attr( $this->get_field_name( 'ogv' ) ); ?>" type="hidden" data-property="ogv" class="media-widget-instance-property" value="<?php echo esc_attr( $ogv ); ?>">
			<input id="<?php echo esc_attr( $this->get_field_id( 'flv' ) ); ?>" name="<?php echo esc_attr( $this->get_field_name( 'flv' ) ); ?>" type="hidden" data-property="flv" class="media-widget-instance-property" value="<?php echo esc_attr( $flv ); ?>">
			<input id="<?php echo esc_attr( $this->get_field_id( 'mov' ) ); ?>" name="<?php echo esc_attr( $this->get_field_name( 'mov' ) ); ?>" type="hidden" data-property="mov" class="media-widget-instance-property" value="<?php echo esc_attr( $mov ); ?>">
			<input id="<?php echo esc_attr( $this->get_field_id( 'quicktime' ) ); ?>" name="<?php echo esc_attr( $this->get_field_name( 'quicktime' ) ); ?>" type="hidden" data-property="quicktime" class="media-widget-instance-property" value="<?php echo esc_attr( $quicktime ); ?>">
			<input id="<?php echo esc_attr( $this->get_field_id( 'attachment_id' ) ); ?>" name="<?php echo esc_attr( $this->get_field_name( 'attachment_id' ) ); ?>" type="hidden" data-property="attachment_id" class="media-widget-instance-property" value="<?php echo esc_attr( $attachment_id ); ?>">
			<input id="<?php echo esc_attr( $this->get_field_id( 'url' ) ); ?>" name="<?php echo esc_attr( $this->get_field_name( 'url' ) ); ?>" type="hidden" data-property="url" class="media-widget-instance-property" value="<?php echo esc_url( $url ); ?>">
			<input id="<?php echo esc_attr( $this->get_field_id( 'reset_widget' ) ); ?>" name="reset_widget" type="hidden" class="reset_widget" value="0">

		</div>

		<?php
	}


	/**
	 * Enqueue preview scripts.
	 *
	 * These scripts normally are enqueued just-in-time when a video shortcode is used.
	 * In the customizer, however, widgets can be dynamically added and rendered via
	 * selective refresh, and so it is important to unconditionally enqueue them in
	 * case a widget does get added.
	 *
	 * @since 4.8.0
	 */
	public function enqueue_preview_scripts() {
		/** This filter is documented in wp-includes/media.php */
		if ( 'mediaelement' === apply_filters( 'wp_video_shortcode_library', 'mediaelement' ) ) {
			wp_enqueue_style( 'wp-mediaelement' );
			wp_enqueue_script( 'mediaelement-vimeo' );
			wp_enqueue_script( 'wp-mediaelement' );
		}
	}

	/**
	 * Loads the required scripts and styles for the widget control.
	 *
	 * @since 4.8.0
	 */
	public function enqueue_admin_scripts() {

		// Identify permitted video file types
		$video_file_types = array( 'mp4', 'm4v', 'webm', 'ogv', 'flv', 'mov' );

		$user_id = get_current_user_id();
		$per_page = get_user_meta( $user_id, 'media_grid_per_page', true );
		if ( empty( $per_page ) || $per_page < 1 ) {
			$per_page = 80;
		}

		//parent::enqueue_admin_scripts();
		wp_enqueue_style( 'wp-mediaelement' );
		wp_enqueue_script( 'mediaelement-vimeo' );
		wp_enqueue_script( 'wp-mediaelement' );

		wp_enqueue_script( 'media-video-widget' );
		wp_localize_script(
			'media-video-widget',
			'VIDEO_WIDGET',
			array(
				'no_video_selected'          => __( 'No video selected' ),
				'add_video'                  => _x( 'Add Video', 'label for button in the media widget' ),
				'replace_video'              => _x( 'Replace Video', 'label for button in the media widget; should preferably not be longer than ~13 characters long' ),
				'edit_video'                 => _x( 'Edit Video', 'label for button in the media widget; should preferably not be longer than ~13 characters long' ),
				'add_to_widget'              => __( 'Add to Widget' ),
				'insert_from_url'            => __( 'Insert from URL' ),
				'missing_attachment'         => sprintf(
					/* translators: %s: URL to media library. */
					__( 'That file cannot be found. Check your <a href="%s">media library</a> and make sure it was not deleted.' ),
					esc_url( admin_url( 'upload.php' ) )
				),
				'unsupported_file_type'      => __( 'Looks like this is not the correct kind of file. Please link to an appropriate file instead.' ),
				'add_subtitles'              => __( 'Add Subtitles' ),
				'remove_video_source'        => __( 'Remove video source' ),
				'media_library'              => __( 'Media Library' ),
				'video_details'              => __( 'Video Details' ),
				'details_button'             => __( 'Video details' ),
				'wrong_url'                  => __( 'No file exists at that URL' ),
				'insert_from_url'            => __( 'Insert from URL' ),
				'remove_video_source'        => __( 'Remove video source' ),
				'cancel_edit'                => __( 'Cancel edit' ),
				'save'                       => __( 'Save' ),
				'media_items'                => __( 'media items' ),
				'includes_url'               => includes_url(),
				'per_page'                   => $per_page,
				'video_file_types'           => $video_file_types,
				'of'                         => __( 'of' ),
			)
		);
	}
}

/**
 * Renders the template for the modal content for the media video widget
 *
 * @since CP-2.5.0
 *
 * @return string
 */
function cp_render_media_video_template() {
	ob_start();
	?>

	<template id="tmpl-edit-video-modal">
		<div id="video-modal-content" style="padding: 2em;">
			<div class="modal-video-details">
				<div class="video-embed" style="max-width:640px;">
					<video class="wp_video_shortcode" controls="" style="width:100%;">
						<source src="<?php echo esc_url( includes_url() . 'js/mediaelement/blank.mp4' ); ?>" type="video/mp4">
					</video>
					<div class="setting" style="margin-top: 1em;">
						<label for="video-details-source" class="name">MP4</label>
						<input type="text" id="video-details-source" readonly="" data-setting="mp4" value="" style="width:100%;">
					</div>
					<button type="button" class="button-link remove-setting" style="display:block;color: #a00; padding: 5px 0;"><?php esc_html_e( 'Remove video source' ); ?></button>

					<fieldset class="setting-group setting preload" style="margin: 1em 0;display: flex;">						
						<label for="preload" class="name" style="margin: 6px 1em 0 0;"><?php esc_html_e( 'Preload' ); ?></label>
						<select id="preload" name="link-type" data-setting="preload">
							<option value="auto"><?php esc_html_e( 'Auto' ); ?></option>
							<option value="metadata"><?php esc_html_e( 'Metadata' ); ?></option>
							<option value="" selected><?php esc_html_e( 'None' ); ?></option>
						</select>
					</fieldset>

					<div class="setting-group" style="margin: 1em 0;display: flex;">
						<div class="setting checkbox-setting">
							<input type="checkbox" id="video-details-loop" data-setting="loop">
							<label for="video-details-loop" class="checkbox-label"><?php esc_html_e( 'Loop' ); ?></label>
						</div>
					</div>

					<div class="setting" data-setting="content">
						<span class="name"><?php esc_html_e( 'Tracks (subtitles, captions, descriptions, chapters, or metadata)' ); ?></span><br>
						<em><?php esc_html_e( 'There are no associated subtitles.' ); ?></em>
						<textarea class="hidden content-setting"></textarea>
					</div>

				</div>
			</div>
		</div>

		<footer class="widget-modal-footer">
			<div class="widget-modal-footer-buttons" style="padding-right: 2em;">
				<button id="video-button-update" type="button" class="button media-button button-primary button-large media-button-select" disabled><?php esc_html_e( 'Update' ); ?></button>
			</div>
		</footer>
	</template>

	<?php
	return ob_get_clean();
}
